function pf_tracking
% This function simulates the Minerva Experiment. It is an omnidirectional
% robot that can look a the ceiling of the Smithsonian and try and localize
% itself. 


% define global variables (can be seen by all functions)
global xVehicleTrue;
global UTrue;
global RTrue;
global ceiling_image;
global nParticles;

%Define Which plots to draw (See plotting section)
drawfig=[1 0 0 0];
drawfig=[0 0 0 1];

%load ceiling image
ceiling_image = imread('ceiling.png');
ceiling_image=ceiling_image(:,:,1);

%initial vehicle state
xVehicleTrue= [340;150;-pi/2];
% Actual process noise used to propagate dynamics
UTrue = diag([0.01,0.01,1.5*pi/180]).^2;
% process noise
RTrue = 3.^2;

% how many particles are we going to use
nParticles=10000;
% The "guess" of what the process noise iss
UEst = 2.0*UTrue;

%Initial Distrubution of Particles -"guess" of where vehicle is
% this is a bunch of particles at the "true" location + noise 1000*Uest
Noise = sqrt(1000*UEst)*randn(3,nParticles);
uNoise=tcomp_batch([0;0;0],Noise);
xVehicleParticles = repmat(xVehicleTrue,1,nParticles);
xVehicleParticles = tcomp_batch(xVehicleParticles,uNoise);

%We can instead start with a bunch of particles spread everywhere- i.e. we
%dont know where the robot is! (uncomment the following
% nParticles=100000;
% Noise = sqrt(10000000*UEst)*randn(3,nParticles);
% xVehicleParticles = repmat(xVehicleTrue,1,nParticles);
% xVehicleParticles = tcomp_batch(xVehicleParticles,Noise);


%length of experiment
nSteps = 450;

%initalize odometry
xOdomLast = GetOdometry(1);

%initalize figures
figure(1)
clf
imshow(ceiling_image);
axis on
hold on

%% START PARTICLE FILTER (for nSteps)
for k = 2:nSteps
    
    %Simulate the vehicle moving forward. This will update the 
    % "ground truth" and create the vehicle odometry
    xOdomNow = SimulateRobot(k);
    
    %We now want to do propogation of particles based on Odometry
    u = tcomp(tinv(xOdomLast),xOdomNow);
    xOdomLast = xOdomNow;
    xVehicleParticles = MotionModelParticles(xVehicleParticles,u,UEst);


    
    %Get "True Measurement" of ceiling intensity z
    z= GetObservation();
    
    
    %Determine "weights" - the observation likelihood p(z_k|x_k) 
    % h is observation (ceiling intenstiy) prediction conditioned on vehicle state
    h = ObservationModel(xVehicleParticles);
    nu = z - h;
    sigma = 10.0;
    weight = (2*pi*sigma^2)^-0.5 * exp(-(nu.^2./(2*sigma^2)));
    
      
    % Do particle Resampling;
    xVehicleParticles = ResampleParticles(xVehicleParticles,weight);
    
    % Do Plotting
    if (drawfig(1)==1)
        figure(1)
        clf
        imshow(ceiling_image);
        axis on
        hold on
        plot(xVehicleParticles(1,:),xVehicleParticles(2,:),'g+')
        plot(xVehicleTrue(1),xVehicleTrue(2),'w+')
        %DoVehicleGraphics(xVehicleEst,eye(3),3,[0 1]);
        drawnow
    end
    if (drawfig(2)==1)
    %PLOT measurement and Predicted Measurements   
    figure(2)
    subplot(2,1,1)
    imshow(uint8(z*ones(50,50)));
    ylabel('observation')
    subplot(2,1,2)
    %imshow(reshape(uint8(h),10,10));  %need to change size based on n particles
    ylabel('prediction')
    drawnow
    end
    
    if (drawfig(3)==1)   
    % A TRACE of the particles  (memory heavy)
    figure(3);
    hold on;
    plot(xVehicleParticles(1,:),xVehicleParticles(2,:),'g+');
    plot(xVehicleTrue(1),xVehicleTrue(2),'k.');
    drawnow;
    hold off;
    end
    
    if (drawfig(4)==1)   
    % Show distribution of particles & truth  
    figure(4);
    subplot(1,2,1)
    plot(xVehicleParticles(1,:),xVehicleParticles(2,:),'g+');
    hold on;
    plot(xVehicleTrue(1),xVehicleTrue(2),'k.');
    hold off;
    subplot(1,2,2)
    plot(zeros(length(xVehicleParticles),1),xVehicleParticles(3,:),'g+');
    axis([-1 1 -pi pi])
    hold on; plot(0,xVehicleTrue(3),'k.'); hold off;
    drawnow;
    end
    
end
    

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function xVehicleParticlesNew = ResampleParticles(xVehicleParticles,weight)
% This is a resampler. You should write your own!
% FUNCTION: Resample particles  (including those removed which left the image)
% INPUT:    xVehicleParticles   (x,y,theta) - location and angle
%           weight              (w)         - particle weight
% OUTPUT:   xVehicleParticlesNew (x,y,theta), with uniform weight
global nParticles;

cumulative = [0 cumsum(weight)];
r = rand(1,nParticles)*cumulative(end);

[n idx] =histc(r,cumulative);

xVehicleParticlesNew=xVehicleParticles(:,idx);




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function xVehicleParticlesNew = MotionModelParticles(xVehicleParticles,u,UEst)

nParticles = size(xVehicleParticles,2);

Noise = sqrt(UEst)*randn(3,nParticles);

uNoise=tcomp_batch(u,Noise);

xVehicleParticlesNew=tcomp_batch(xVehicleParticles,uNoise);



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function tac = tcomp_batch(tab,tbc)
% This function propogates the state
% tbc is control input = [v_x, v_y, v_theta]
% tab is current state = [x, y, theta]

theta_new = tab(3,:)+tbc(3,:);
theta_new = AngleWrap(theta_new);

s = sin(tab(3,:));
c = cos(tab(3,:));

tac = [tab(1,:) + c .* tbc(1,:) - s .* tbc(2,:);
       tab(2,:) + s .* tbc(1,:) + c .* tbc(2,:);
       theta_new];


    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function zMeasured = GetObservation()
global xVehicleTrue;
global RTrue;

zTrue = ObservationModel(xVehicleTrue);
Noise = sqrt(RTrue)*randn(1);

zMeasured = zTrue + Noise;



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function h = ObservationModel(xVehicle)
global ceiling_image;

c=round(xVehicle(1,:));
r=round(xVehicle(2,:));
[height width] = size(ceiling_image);
c=min(max(1,c),width);
r=min(max(1,r),height);
idx = r+(c-1)*height;
h = double(ceiling_image(idx));


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [Odometry] = SimulateRobot(k)
global xVehicleTrue;
% This is the defined robot control
u = GetRobotControl(k);
% This propogates the vehicle forward
xVehicleTrue = tcomp(xVehicleTrue,u);
% Now this gets noise added to it- and generates odometry
Odometry = GetOdometry(k);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [xnow] = GetOdometry(k)
persistent LastOdom; %internal to robot low-level controller
global UTrue;
if(isempty(LastOdom))
    global xVehicleTrue;
    LastOdom = xVehicleTrue;
end;
u = GetRobotControl(k);
xnow =tcomp(LastOdom,u);
uNoise = sqrt(UTrue)*randn(3,1);  % add noise into system
xnow = tcomp(xnow,uNoise);
LastOdom = xnow;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function u = GetRobotControl(k)
u = [0; 1.3 ; 0.9*pi/180];
